ColorConfiguratorPickerElement = {}

local ColorConfiguratorPickerElement_mt = Class(ColorConfiguratorPickerElement, BitmapElement)

function ColorConfiguratorPickerElement.new(target, custom_mt)
	local self = BitmapElement.new(target, custom_mt or ColorConfiguratorPickerElement_mt)

	self.background = nil
	self.pointer = nil
	self.slider = nil

	self.mouseDown = false

	self.minValue = 0
	self.maxValue = 100

	self.hue = 0
	self.saturation = 0
	self.value = 100

	self.minAbsPointerPosX = 0
	self.maxAbsPointerPosX = 0
	self.minAbsPointerPosY = 0
	self.maxAbsPointerPosY = 0

	return self
end

function ColorConfiguratorPickerElement:loadFromXML(xmlFile, key)
	ColorConfiguratorPickerElement:superClass().loadFromXML(self, xmlFile, key)

	self:addCallback(xmlFile, key .. "#onChanged", "onChangedCallback")
	self:addCallback(xmlFile, key .. "#onFocus", "onFocusCallback")
	self:addCallback(xmlFile, key .. "#onLeave", "onLeaveCallback")
end

function ColorConfiguratorPickerElement:onGuiSetupFinished()
	ColorConfiguratorPickerElement:superClass().onGuiSetupFinished(self)

	self.background = self:getDescendantByName("pickerBackground")
	self.pointer = self:getDescendantByName("pickerPointer")
	self.slider = self:getDescendantByName("pickerSlider")

	ColorConfiguratorUtil.appendedFunction(self.slider, "updateSliderPosition", self, "slider_updateSliderPosition")
	ColorConfiguratorUtil.appendedFunction(self.slider, "callOnChanged", self, "slider_callOnChanged")
	ColorConfiguratorUtil.overwrittenFunction(self.slider, "shouldFocusChange", self, "slider_shouldFocusChange")
end

function ColorConfiguratorPickerElement:slider_updateSliderPosition(slider)
	slider.sliderPosition = {GuiUtils.alignToScreenPixels(unpack(slider.sliderPosition))}
end

function ColorConfiguratorPickerElement:slider_callOnChanged(slider)
	local hue = slider.maxValue - slider.currentValue
	local r, g, b = ColorConfiguratorUtil.hueToRgb(hue)

	self.background:setImageColor(nil, ColorConfiguratorUtil.rgbToGiants(r), ColorConfiguratorUtil.rgbToGiants(g), ColorConfiguratorUtil.rgbToGiants(b), 1)
	self.hue = hue

	self:callOnChanged(true)
end

function ColorConfiguratorPickerElement:getIsActive()
	local baseActive = ColorConfiguratorPickerElement:superClass().getIsActive(self)

	return baseActive and self.background ~= nil and self.pointer ~= nil and self.slider ~= nil and not self.target.sceneRender.mouseDown
end

function ColorConfiguratorPickerElement:updatePointerLimits()
	local background = self.background
	local pointer = self.pointer

	if background ~= nil and pointer ~= nil then
		self.minAbsPointerPosX = background.absPosition[1] - pointer.absSize[1] * 0.5
		self.maxAbsPointerPosX = background.absPosition[1] + background.absSize[1] - pointer.absSize[1] * 0.5
		self.minAbsPointerPosY = background.absPosition[2] - pointer.absSize[2] * 0.5
		self.maxAbsPointerPosY = background.absPosition[2] + background.absSize[2] - pointer.absSize[2] * 0.5
	end
end

function ColorConfiguratorPickerElement:mouseEvent(posX, posY, isDown, isUp, button, eventUsed)
	if self:getIsActive() then
		if ColorConfiguratorPickerElement:superClass().mouseEvent(self, posX, posY, isDown, isUp, button, eventUsed) then
			eventUsed = true
		end

		if self.mouseDown and isUp and button == Input.MOUSE_BUTTON_LEFT then
			eventUsed = true
			self.clickedOnPointer = false
			self.mouseDown = false
		end

		if not eventUsed and (GuiUtils.checkOverlayOverlap(posX, posY, self.background.absPosition[1], self.background.absPosition[2], self.background.absSize[1], self.background.absSize[2]) or GuiUtils.checkOverlayOverlap(posX, posY, self.pointer.absPosition[1], self.pointer.absPosition[2], self.pointer.absSize[1], self.pointer.absSize[2])) then
			eventUsed = true

			if isDown and button == Input.MOUSE_BUTTON_LEFT then
				if not self.mouseDown and GuiUtils.checkOverlayOverlap(posX, posY, self.pointer.absPosition[1], self.pointer.absPosition[2], self.pointer.absSize[1], self.pointer.absSize[2]) then
					self.clickedOnPointer = true
					self.lastMousePosX = posX
					self.lastMousePosY = posY
					self.lastPointerPosX = self.pointer.absPosition[1]
					self.lastPointerPosY = self.pointer.absPosition[2]
				end

				self.mouseDown = true
			end
		end

		if self.mouseDown then
			eventUsed = true

			local mousePosX = posX
			local mousePosY = posY

			if self.clickedOnPointer then
				local deltaX = posX - self.lastMousePosX
				local deltaY = posY - self.lastMousePosY

				mousePosX = self.lastPointerPosX + deltaX
				mousePosY = self.lastPointerPosY + deltaY
			else
				mousePosX = mousePosX - self.pointer.absSize[1] * 0.5
				mousePosY = mousePosY - self.pointer.absSize[2] * 0.5
			end

			local newValueX = self.minValue + (mousePosX - self.minAbsPointerPosX) / (self.maxAbsPointerPosX - self.minAbsPointerPosX) * (self.maxValue - self.minValue)
			local newValueY = self.minValue + (1 - (mousePosY - self.minAbsPointerPosY) / (self.maxAbsPointerPosY - self.minAbsPointerPosY)) * (self.maxValue - self.minValue)

			self:setValue(newValueX, newValueY)
		end

		local cursorInPicker = GuiUtils.checkOverlayOverlap(posX, posY, self.absPosition[1], self.absPosition[2], self.absSize[1], self.absSize[2])
		local cursorInSlider = GuiUtils.checkOverlayOverlap(posX, posY, self.slider.absPosition[1], self.slider.absPosition[2], self.slider.absSize[1], self.slider.absSize[2])

		if cursorInPicker or cursorInSlider then
			if isDown and (cursorInPicker and button == Input.MOUSE_BUTTON_LEFT) or (cursorInSlider and (Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_DOWN) or Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_UP))) then
				if self.handleFocus and self:getOverlayState() ~= GuiOverlay.STATE_FOCUSED then
					FocusManager:setFocus(self)
				end
			end

			if not isUp and not self.mouseEntered then
				if self:getOverlayState() == GuiOverlay.STATE_NORMAL then
					FocusManager:setHighlight(self)
				end

				self.mouseEntered = true
			end
		else
			self:restoreOverlayState()
			self.mouseEntered = false

			FocusManager:unsetHighlight(self)
		end
	end

	return eventUsed
end

function ColorConfiguratorPickerElement:setValue(newValueX, newValueY)
	newValueX = math.min(math.max(newValueX, self.minValue), self.maxValue)
	newValueY = math.min(math.max(newValueY, self.minValue), self.maxValue)

	local stateX = (newValueX - self.minValue) / (self.maxValue - self.minValue)
	local stateY = (newValueY - self.minValue) / (self.maxValue - self.minValue)
	local xPos, yPos = MathUtil.lerp(self.minAbsPointerPosX, self.maxAbsPointerPosX, stateX), MathUtil.lerp(self.minAbsPointerPosY, self.maxAbsPointerPosY, 1 - stateY)

	self.pointer:setAbsolutePosition(GuiUtils.alignToScreenPixels(xPos, yPos))

	newValueY = MathUtil.round(self.maxValue - newValueY)
	newValueX = MathUtil.round(newValueX)

	if newValueX ~= self.saturation or newValueY ~= self.value then
		self.saturation = newValueX
		self.value = newValueY

		self:callOnChanged(false)

		return true
	end

	return false
end

function ColorConfiguratorPickerElement:setValues(hsvColors)
	local h, s, v = unpack(hsvColors)

	h = h or self.hue
	s = s or self.saturation
	v = v or self.value

	self.slider:setValue(self.slider.maxValue - h)
	self:setValue(s, self.maxValue - v)
end

function ColorConfiguratorPickerElement:getValues()
	return self.hue, self.saturation, self.value
end

function ColorConfiguratorPickerElement:getFocusTarget(incomingDirection, moveDirection)
	return self
end

function ColorConfiguratorPickerElement:canReceiveFocus()
	return not self:getIsDisabled() and self:getIsVisible() and self:getHandleFocus()
end

function ColorConfiguratorPickerElement:onFocusLeave()
	self:raiseCallback("onLeaveCallback", self)
end

function ColorConfiguratorPickerElement:onFocusEnter()
	self:raiseCallback("onFocusCallback", self)
end

function ColorConfiguratorPickerElement:slider_shouldFocusChange(slider, superFunc, direction)
	return true
end

function ColorConfiguratorPickerElement:callOnChanged(isSliderChange)
	local h, s, v = self:getValues()
	local r, g, b = ColorConfiguratorUtil.hsvToGiants(h, s, v)

	self.pointer:setImageColor(nil, r, g, b, 1)
	self:raiseCallback("onChangedCallback", {h, s, v}, isSliderChange)
end