ColorConfiguratorDialog = {
	CONTROLS = {
		"titleText",
		"multiMaterial",
		"textInputRed",
		"textInputGreen",
		"textInputBlue",
		"sliderRed",
		"sliderGreen",
		"sliderBlue",
		"sceneRender",
		"tooltipLayout",
		"glyphBox",
		"saveButton",
		"multiMode",
		"colorPicker",
		"textInputHue",
		"textInputSaturation",
		"textInputValue",
		"colorButton",
		"zoomGlyph",
		"zoomGlyphText",
		"lookGlyph",
		"lookGlyphText",
		"dialogButtonLayout"
	},
	COLOR = {
		RED = 1,
		GREEN = 2,
		BLUE = 3,
		MATERIAL = 4
	},
	L10N_SYMBOL = {
		COPY_COLOR = "colorConfigurator_toolTip_colorCopied",
		PASTE_COLOR = "colorConfigurator_toolTip_colorPasted",
		PASTE_COLOR_CLIPBOARD = "colorConfigurator_toolTip_colorPastedClipboard",
		MISSING_MATERIAL = "colorConfigurator_toolTip_missingMaterial",
		COLOR_MODE = "colorConfigurator_button_mode",
		MODE_GIANTS = "colorConfigurator_ui_mode_giants",
		MODE_RGB = "colorConfigurator_ui_mode_rgb",
		ALREADY_EXISTS = "colorConfigurator_ui_alreadyExists",
		LIMIT_REACHED = "colorConfigurator_ui_limitReached",
		ADDED_COLOR = "colorConfigurator_ui_colorAddedToList"
	},
	MODE = {
		GIANTS = 1,
		RGB = 2
	},
	MAX_VEHICLE_CONFIGS = 128,
	MAX_GLOBAL_CONFIGS = 64
}

local ColorConfiguratorDialog_mt = Class(ColorConfiguratorDialog, MessageDialog)

local function NO_CALLBACK()
end

function ColorConfiguratorDialog.new(target, custom_mt, materialsXMLFilename, colorConfigurator)
	local self = MessageDialog.new(target, custom_mt or ColorConfiguratorDialog_mt)

	self:registerControls(ColorConfiguratorDialog.CONTROLS)

	self.colorConfigurator = colorConfigurator
	self.pickerDialog = colorConfigurator.pickerDialog

	self.materials = self:loadMaterials(materialsXMLFilename, false)
	self.colors = {1, 1, 1, 0}

	self.isRGBMode = colorConfigurator:getSetting("isRGBMode")

	self.lastInputMode = nil
	self.lastInputHelpMode = nil

	self.copiedColors = nil

	self.sliderSample = GuiSoundPlayer.SOUND_SAMPLES.HOVER
	self.textSample = nil

	self.materialAvailable = true

	self.eventIdUpDownController = ""
	self.eventIdLeftRightController = ""

	self.pickerSelected = false

	self.disabledMaterialConfigs = {
		["wrappingColor"] = {
			"baleWrapper",
			"balerWrapper",
			"inlineWrapper",
			"pdlc_goeweilPack.baleWrapperExtended",
			"pdlc_goeweilPack.balerWrapperExtended",
			"pdlc_goeweilPack.balerStationary",
		},
		["baseColor"] = {
			"pdlc_pumpsAndHosesPack.hosePallet"
		}
	}

	return self
end

function ColorConfiguratorDialog:loadMaterials(xmlFilename, printToLog)
	local materials = {}
	local xmlFile = XMLFile.load("materialsXML", xmlFilename)

	if xmlFile ~= nil then
		xmlFile:iterate("materials.material", function (_, key)
			local name = xmlFile:getString(key .. "#name")
			local shaderId = xmlFile:getInt(key .. "#shaderId")

			if name ~= nil and name:len() > 0 and shaderId ~= nil then
				materials[#materials + 1] = {
					name = name,
					shaderId = shaderId
				}
			end
		end)

		xmlFile:delete()

		ColorConfiguratorUtil.printOnLoad(printToLog, xmlFile)
	end

	return materials
end

function ColorConfiguratorDialog:onMapLoad()
	g_i3DManager:pinSharedI3DFileInCache(self.sceneRender.filename)
end

function ColorConfiguratorDialog:onGuiSetupFinished()
	ColorConfiguratorDialog:superClass().onGuiSetupFinished(self)

	self.textInputElements = {
		self.textInputRed,
		self.textInputGreen,
		self.textInputBlue
	}

	self.textInputHSVElements = {
		self.textInputHue,
		self.textInputSaturation,
		self.textInputValue
	}

	self.sliderElements = {
		self.sliderRed,
		self.sliderGreen,
		self.sliderBlue
	}

	for _, elem in pairs(self.textInputElements) do
		elem.customFocusSample = self.textSample
	end

	for _, elem in pairs(self.textInputHSVElements) do
		elem.customFocusSample = self.textSample
	end

	local materialsTexts = {}

	for i, mat in pairs(self.materials) do
		table.insert(materialsTexts, string.format("%s (%d)", g_i18n:getText(string.format("colorConfigurator_material_%s", mat.name)), i))
	end

	self.multiMaterial:setTexts(materialsTexts)

	self.multiMode:setTexts({
		g_i18n:getText(ColorConfiguratorDialog.L10N_SYMBOL.MODE_GIANTS),
		g_i18n:getText(ColorConfiguratorDialog.L10N_SYMBOL.MODE_RGB)
	})

	self.lookGlyphText:setText(g_i18n:getText("colorConfigurator_ui_colorS") .. "/" .. g_i18n:getText("colorConfigurator_ui_colorV"))
	self.dialogButtonLayout:invalidateLayout()

	ColorConfiguratorUtil.overwrittenFunction(self.colorButton, "getIsActive", self, "colorButton_getIsActive")

	self.sceneRender.filename = string.gsub(self.sceneRender.filename, "$moddir$/", self.colorConfigurator.baseDirectory)

	self:setRGBMode(self.isRGBMode)
end

function ColorConfiguratorDialog:onCreate(element)
	ColorConfiguratorDialog:superClass().onCreate(self, element)

	self.colorPicker:updatePointerLimits()
	self:updateHSVElements()
end

function ColorConfiguratorDialog:colorButton_getIsActive(button, superFunc)
	local baseActive = superFunc(button)

	return baseActive and not self.sceneRender.mouseDown
end

function ColorConfiguratorDialog:update(dt)
	ColorConfiguratorDialog:superClass().update(self, dt)

	local currentInputMode = g_inputBinding:getLastInputMode()

	if currentInputMode ~= self.lastInputMode then
		self.lastInputMode = currentInputMode

		local isController = currentInputMode == GS_INPUT_HELP_MODE_GAMEPAD

		g_inputBinding:setActionEventActive(self.eventIdUpDownController, isController)
		g_inputBinding:setActionEventActive(self.eventIdLeftRightController, isController)
	end

	local inputHelpMode = g_inputBinding:getInputHelpMode()

	if inputHelpMode ~= self.lastInputHelpMode then
		self.lastInputHelpMode = inputHelpMode

		local isController = inputHelpMode == GS_INPUT_HELP_MODE_GAMEPAD

		self:updateInputGlyphsVisibility(isController and self.pickerSelected)

		local actions1 = nil
		local actions2 = nil
		local actions3 = nil

		if isController then
			actions1 = {
				InputAction.AXIS_MAP_ZOOM_IN
			}
			actions2 = {
				InputAction.AXIS_MAP_ZOOM_OUT
			}
			actions3 = {
				InputAction.AXIS_LOOK_UPDOWN_VEHICLE,
				InputAction.AXIS_LOOK_LEFTRIGHT_VEHICLE
			}
		else
			actions1 = {
				InputAction.AXIS_MAP_ZOOM_IN,
				InputAction.AXIS_MAP_ZOOM_OUT
			}
			actions2 = {
				InputAction.AXIS_LOOK_LEFTRIGHT_DRAG,
				InputAction.AXIS_LOOK_UPDOWN_DRAG
			}
			actions3 = {
				InputAction.MOUSE_ALT_COMMAND3_BUTTON
			}
		end

		self.glyphBox.elements[1]:setActions(actions1)
		self.glyphBox.elements[2]:setActions(actions2)
		self.glyphBox.elements[3]:setActions(actions3)

		self.zoomGlyph:setActions({
			InputAction.AXIS_MAP_ZOOM_IN,
			InputAction.AXIS_MAP_ZOOM_OUT
		})

		self.lookGlyph:setActions({
			InputAction.AXIS_LOOK_LEFTRIGHT_VEHICLE,
			InputAction.AXIS_LOOK_UPDOWN_VEHICLE
		})
	end
end

function ColorConfiguratorDialog:updateInputGlyphsVisibility(visible)
	self.zoomGlyph:setVisible(visible)
	self.zoomGlyphText:setVisible(visible)
	self.lookGlyph:setVisible(visible)
	self.lookGlyphText:setVisible(visible)
	self.dialogButtonLayout:invalidateLayout()
end

function ColorConfiguratorDialog:registerActionEvents()
	g_inputBinding:registerActionEvent(InputAction.COLOR_CONFIGURATOR_COPY, self, self.onInputCopy, false, true, false, true)
	g_inputBinding:registerActionEvent(InputAction.COLOR_CONFIGURATOR_PASTE, self, self.onInputPaste, false, true, false, true)
	g_inputBinding:registerActionEvent(InputAction.COLOR_CONFIGURATOR_PASTE_FROM_CLIPBOARD, self, self.onInputPasteFromClipboard, false, true, false, true)

	local isController = g_inputBinding:getLastInputMode() == GS_INPUT_HELP_MODE_GAMEPAD
	local _ = nil

	_, self.eventIdUpDownController = g_inputBinding:registerActionEvent(InputAction.AXIS_LOOK_UPDOWN_VEHICLE, self, self.onUpDownController, false, false, true, isController)
	_, self.eventIdLeftRightController = g_inputBinding:registerActionEvent(InputAction.AXIS_LOOK_LEFTRIGHT_VEHICLE, self, self.onLeftRightController, false, false, true, isController)

	g_inputBinding:registerActionEvent(InputAction.AXIS_MAP_ZOOM_IN, self, self.onZoomController, false, false, true, true, -1)
	g_inputBinding:registerActionEvent(InputAction.AXIS_MAP_ZOOM_OUT, self, self.onZoomController, false, false, true, true, 1)
end

function ColorConfiguratorDialog:removeActionEvents()
	g_inputBinding:removeActionEventsByTarget(self)
end

function ColorConfiguratorDialog:onClickOk()
	self:close()

	self.colorConfigurator:setColor(self.colors)
	self.colorConfigurator:addTempConfig(self.colors, self:getCurrentItemName(true))
end

function ColorConfiguratorDialog:onSaveConfig(configText, isVehicleConfig)
	local alreadyExists, limitReached = self:getCanAddConfig(isVehicleConfig)

	if limitReached or alreadyExists then
		local dialogText = ColorConfiguratorDialog.L10N_SYMBOL.ALREADY_EXISTS

		if limitReached then
			dialogText = ColorConfiguratorDialog.L10N_SYMBOL.LIMIT_REACHED
		end

		g_gui:showInfoDialog({
			text = g_i18n:getText(dialogText),
			dialogType = DialogElement.TYPE_WARNING,
		})

		return
	end

	local r, g, b, mat = unpack(self.colors)
	local data = {
		name = configText,
		material = mat,
		color = {
			r,
			g,
			b
		}
	}

	if isVehicleConfig then
		data.isAddConfig = true

		local storeItem = self.pickerDialog.target.storeItem
		local configName = self.pickerDialog.args.configName

		if storeItem.bundleInfo ~= nil then
			for _, bundleItem in ipairs(storeItem.bundleInfo.bundleItems) do
				self.colorConfigurator:addAdditionalConfig(bundleItem.xmlFilename, configName, data, true)
			end
		end

		self.colorConfigurator:addAdditionalConfig(storeItem.xmlFilename, configName, data)
	else
		self.colorConfigurator:addBaseConfig(data)
	end

	g_gui:showInfoDialog({
		text = string.format(g_i18n:getText(ColorConfiguratorDialog.L10N_SYMBOL.ADDED_COLOR), data.name),
		dialogType = DialogElement.TYPE_INFO
	})
end

function ColorConfiguratorDialog:onClickSave()
	for _, elem in pairs(self.textInputElements) do
		if elem:getOverlayState() == GuiOverlay.STATE_PRESSED then
			return
		end
	end

	for _, elem in pairs(self.textInputHSVElements) do
		if elem:getOverlayState() == GuiOverlay.STATE_PRESSED then
			return
		end
	end

	g_gui:showDialog("ColorConfiguratorSaveDialog")
end

function ColorConfiguratorDialog:onClickBack()
	ColorConfiguratorDialog:superClass().onClickBack(self)

	local pickerDialog = self.pickerDialog

	pickerDialog.disableOpenSound = true

	g_gui:showDialog(pickerDialog.name)

	pickerDialog:setColors(unpack(pickerDialog.previousColors))
	pickerDialog.disableOpenSound = false
end

function ColorConfiguratorDialog:onClickColorMode(state)
	local isRGBMode = state == ColorConfiguratorDialog.MODE.RGB

	self:setRGBMode(isRGBMode)
end

function ColorConfiguratorDialog:onOpen()
	ColorConfiguratorDialog:superClass().onOpen(self)

	self.sceneRender:createScene()
	self:registerActionEvents()

	self.materialAvailable = self:getIsMaterialAvailable()
end

function ColorConfiguratorDialog:initializeScreen()
	ColorConfiguratorDialog:superClass().initializeScreen(self)

	self:setSoundSuppressed(true)
	FocusManager:setFocus(self.sliderRed.parent)
	self:setSoundSuppressed(false)
end

function ColorConfiguratorDialog:onClose()
	self:removeActionEvents()

	self.sceneRender:destroyScene()
	self:updateTooltipLayout(false)

	self.colorConfigurator:updateSetting("isRGBMode", self.isRGBMode)

	ColorConfiguratorDialog:superClass().onClose(self)
end

function ColorConfiguratorDialog:onRenderLoad(scene, overlay)
	self:updateColorPreview()
end

function ColorConfiguratorDialog:onClickMaterialType(state)
	self.colors[ColorConfiguratorDialog.COLOR.MATERIAL] = self.materials[state].shaderId

	self:updateColorPreview()
	self:updateTooltipLayout(false)
end

function ColorConfiguratorDialog:onButtonPressed(element, dismissal)
	self:verifyInputText(element, dismissal)
end

function ColorConfiguratorDialog:onTextChangedRed(element)
	self:onRGBElementChanged(ColorConfiguratorDialog.COLOR.RED, element.text, true)
end

function ColorConfiguratorDialog:onTextChangedGreen(element)
	self:onRGBElementChanged(ColorConfiguratorDialog.COLOR.GREEN, element.text, true)
end

function ColorConfiguratorDialog:onTextChangedBlue(element)
	self:onRGBElementChanged(ColorConfiguratorDialog.COLOR.BLUE, element.text, true)
end

function ColorConfiguratorDialog:onChangedSliderRed(value)
	self:onRGBElementChanged(ColorConfiguratorDialog.COLOR.RED, value, false)
end

function ColorConfiguratorDialog:onChangedSliderGreen(value)
	self:onRGBElementChanged(ColorConfiguratorDialog.COLOR.GREEN, value, false)
end

function ColorConfiguratorDialog:onChangedSliderBlue(value)
	self:onRGBElementChanged(ColorConfiguratorDialog.COLOR.BLUE, value, false)
end

function ColorConfiguratorDialog:onIsUnicodeAllowedTextRed(unicode)
	return self:onIsUnicodeAllowed(self.textInputRed, unicode)
end

function ColorConfiguratorDialog:onIsUnicodeAllowedTextGreen(unicode)
	return self:onIsUnicodeAllowed(self.textInputGreen, unicode)
end

function ColorConfiguratorDialog:onIsUnicodeAllowedTextBlue(unicode)
	return self:onIsUnicodeAllowed(self.textInputBlue, unicode)
end

function ColorConfiguratorDialog:onRGBElementChanged(id, value, textChanged, forceUpdate)
	local colorValue = tonumber(value) or 0
	local sliderElement = self.sliderElements[id]
	local textInputElement = self.textInputElements[id]

	if not forceUpdate then
		if textChanged then
			self:setSliderValueWithNoCallback(sliderElement, colorValue)
		else
			self:setTextWithNoCallback(textInputElement, tostring(colorValue))
			self:playSample(self.sliderSample)
		end
	else
		self:setSliderValueWithNoCallback(sliderElement, colorValue)
		self:setTextWithNoCallback(textInputElement, tostring(colorValue))
	end

	if self.isRGBMode then
		colorValue = ColorConfiguratorUtil.rgbToGiants(colorValue)
	end

	self.colors[id] = colorValue

	if not forceUpdate then
		self:updateHSVElements()
		self:updateColorPreview()
		self:updateTooltipLayout(false)
	end
end

function ColorConfiguratorDialog:getTypedText(element, unicode)
	local text = (element.cursorPosition > 1 and utf8Substr(element.text, 0, element.cursorPosition - 1) or "") .. unicodeToUtf8(unicode) .. (element.cursorPosition <= utf8Strlen(element.text) and utf8Substr(element.text, element.cursorPosition - 1) or "")
	local number = tonumber(text)

	return text, number
end

function ColorConfiguratorDialog:onIsUnicodeAllowed(element, unicode)
	local text, number = self:getTypedText(element, unicode)

	if number == nil or unicode == Input.KEY_space then
		return false
	end

	if self.isRGBMode then
		return not MathUtil.getIsOutOfBounds(number, 0, 255) and not string.startsWith(text, "02") and not string.startsWith(text, "01") and not string.startsWith(text, "00") and unicode ~= Input.KEY_period
	end

	return not MathUtil.getIsOutOfBounds(number, 0, 1) and not string.startsWith(text, "1.") and not string.startsWith(text, "01") and not string.startsWith(text, "00")
end

function ColorConfiguratorDialog:setValues(colors, hasMaterial, isModeChange)
	for id = 1, 3 do
		local color = colors[id]

		if self.isRGBMode then
			color = ColorConfiguratorUtil.giantsToRgb(color)
		end

		self:onRGBElementChanged(id, color, _, true)
	end

	if self.materialAvailable then
		local state = self:getMaterialState(colors[ColorConfiguratorDialog.COLOR.MATERIAL] or 0)

		if self.multiMaterial.state ~= state then
			self.multiMaterial:setState(state, true)
		end
	end

	if not isModeChange then
		self:updateHSVElements()
		self:updateColorPreview()
		self:updateTooltipLayout(not hasMaterial, g_i18n:getText(ColorConfiguratorDialog.L10N_SYMBOL.MISSING_MATERIAL))
	end
end

function ColorConfiguratorDialog:verifyInputText(textInputElement, dismissal)
	local colorNumber = tonumber(textInputElement.text) or 0
	local maxValue = 1

	if self.isRGBMode then
		maxValue = 255
	end

	textInputElement:setText(tostring(MathUtil.clamp(colorNumber, 0, maxValue)))
end

function ColorConfiguratorDialog:updateColorPreview()
	local r, g, b, mat = unpack(self.colors)
	local isMetallic = ConfigurationUtil.isColorMetallic(mat)
	local element = self.colorButton

	element:getDescendantByName("colorImage"):setImageColor(nil, r, g, b, 1)
	element:getDescendantByName("colorImage"):setVisible(not isMetallic)
	element:getDescendantByName("colorImageMetallic"):setImageColor(nil, r, g, b, 1)
	element:getDescendantByName("colorImageMetallic"):setVisible(isMetallic)

	self.sceneRender:updateShaderParameter(r, g, b, mat)
end

function ColorConfiguratorDialog:updateTooltipLayout(visible, text)
	if self.tooltipLayout:getIsVisible() ~= visible then
		self.tooltipLayout:setVisible(visible)
	end

	if text ~= nil then
		self.tooltipLayout.elements[1]:setText(text)
	end
end

function ColorConfiguratorDialog:getCanAddConfig(isVehicleConfig)
	local r, g, b, mat = unpack(self.colors)
	local alreadyExists = false
	local limitReached = false

	if isVehicleConfig then
		local storeItem = self.pickerDialog.target.storeItem
		local configName = self.pickerDialog.args.configName
		local baseConfigs = storeItem.configurations[configName]

		for _, config in pairs(baseConfigs) do
			if not config.isCustomConfig then
				local configMaterial = config.material
				local configColor = config.color

				if mat == configMaterial and ColorConfiguratorUtil.equals(configColor, {r, g, b}) then
					alreadyExists = true
					break
				end
			end
		end

		local addConfigsCount = 0

		if not alreadyExists then
			local filename = storeItem.xmlFilename
			local addConfigs = self.colorConfigurator:getAdditionalConfigs(filename, configName)

			if addConfigs ~= nil then
				addConfigsCount = #addConfigs

				for _, addColor in ipairs(addConfigs) do
					local configMaterial = addColor.material
					local configColor = addColor.color

					if mat == configMaterial and ColorConfiguratorUtil.equals(configColor, {r, g, b}) then
						alreadyExists = true
						break
					end
				end
			end
		end

		limitReached = #baseConfigs + addConfigsCount >= ColorConfiguratorDialog.MAX_VEHICLE_CONFIGS
	else
		local baseConfigs = self.colorConfigurator:getBaseConfigs()

		for _, config in ipairs(baseConfigs) do
			local configMaterial = config.material
			local configColor = config.color

			if mat == configMaterial and ColorConfiguratorUtil.equals(configColor, {r, g, b}) then
				alreadyExists = true
				break
			end
		end

		limitReached = #baseConfigs >= ColorConfiguratorDialog.MAX_GLOBAL_CONFIGS
	end

	return alreadyExists, limitReached
end

function ColorConfiguratorDialog:getMaterialState(material)
	for i, mat in pairs(self.materials) do
		if mat.shaderId == material then
			return i
		end
	end

	return 1
end

function ColorConfiguratorDialog:onInputCopy()
	self.copiedColors = table.copy(self.colors)

	self:updateTooltipLayout(true, g_i18n:getText(ColorConfiguratorDialog.L10N_SYMBOL.COPY_COLOR))
	self:playSample(GuiSoundPlayer.SOUND_SAMPLES.CLICK)
end

function ColorConfiguratorDialog:onInputPaste()
	local colors = self.copiedColors

	if colors ~= nil then
		self:setValues(colors, true)
		self:updateTooltipLayout(true, g_i18n:getText(ColorConfiguratorDialog.L10N_SYMBOL.PASTE_COLOR))
		self:playSample(GuiSoundPlayer.SOUND_SAMPLES.CLICK)
	end
end

function ColorConfiguratorDialog:onInputPasteFromClipboard()
	local text = ColorConfiguratorUtil.getClipboardText()

	if text ~= nil and text ~= "" then
		local values = nil

		if string.startsWith(text, "#") and text:len() == 7 then
			local text = text:sub(2)
			local tbl = {}

			for c in text:gmatch(('.'):rep(2)) do
				local num = tonumber(c, 16)

				if num == nil then
					break
				end

				table.insert(tbl, num)
			end

			if #tbl == 3 then
				values = tbl
			end
		else
			for _, sep in ipairs({",", "-", "/", "\\", "	"}) do
				text = text:gsub(sep, " ")
			end

			local tbl = {}

			for c in text:gmatch("[^ ]+") do
				local num = tonumber(c)

				if num == nil then
					break
				end

				table.insert(tbl, num)
			end

			if #tbl >= 3 then
				values = tbl
			end
		end

		if values ~= nil then
			local colors = {}

			for i = 1, 3 do
				colors[i] = MathUtil.clamp(values[i], 0, self.isRGBMode and 255 or 1)

				if self.isRGBMode then
					colors[i] = ColorConfiguratorUtil.rgbToGiants(colors[i])
				end
			end

			colors[4] = values[4]

			self:setValues(colors, colors[4] ~= nil)
			self:updateTooltipLayout(true, g_i18n:getText(ColorConfiguratorDialog.L10N_SYMBOL.PASTE_COLOR_CLIPBOARD))
			self:playSample(GuiSoundPlayer.SOUND_SAMPLES.CLICK)
		end
	end
end

function ColorConfiguratorDialog:onUpDownController(_, inputValue, _, isAnalog, isMouse)
	if isAnalog and inputValue ~= 0 then
		if self.pickerSelected then
			local h, s, v = self.colorPicker:getValues()

			self.colorPicker:setValues({h, s, v - inputValue})
		else
			self.sceneRender:updateRotation(-inputValue * 0.04, 0)
		end
	end
end

function ColorConfiguratorDialog:onLeftRightController(_, inputValue, _, isAnalog, isMouse)
	if isAnalog and inputValue ~= 0 then
		if self.pickerSelected then
			local h, s, v = self.colorPicker:getValues()

			self.colorPicker:setValues({h, s + inputValue, v})
		else
			self.sceneRender:updateRotation(0, inputValue * 0.04)
		end
	end
end

function ColorConfiguratorDialog:onZoomController(_, inputValue, direction, isAnalog, isMouse)
	if isAnalog and inputValue ~= 0 then
		if self.pickerSelected then
			local h, s, v = self.colorPicker:getValues()

			self.colorPicker:setValues({h - inputValue * direction, s, v})
		else
			self.sceneRender:updateCameraZoom(inputValue * direction * 0.04)
		end
	end
end

function ColorConfiguratorDialog:setRGBMode(isRGBMode)
	self.isRGBMode = isRGBMode

	local sliderMaxValue = 1
	local sliderStepSize = 0.003921568627451
	local multiState = ColorConfiguratorDialog.MODE.GIANTS

	if isRGBMode then
		sliderMaxValue = 255
		sliderStepSize = 1
		multiState = ColorConfiguratorDialog.MODE.RGB
	end

	for _, elem in pairs(self.sliderElements) do
		elem.maxValue = sliderMaxValue
		elem.stepSize = sliderStepSize
	end

	if self.multiMode.state ~= multiState then
		self.multiMode:setState(multiState, true)
	end

	for _, elem in pairs(self.textInputElements) do
		elem:setMinMaxValues(0, sliderMaxValue, sliderStepSize)
	end

	self:setValues(self.colors, true, true)
end

function ColorConfiguratorDialog:setTextWithNoCallback(textInputElement, text)
	local onTextChangedCallbackBackup = textInputElement.onTextChangedCallback

	textInputElement.onTextChangedCallback = NO_CALLBACK
	textInputElement:setText(text)
	textInputElement.onTextChangedCallback = onTextChangedCallbackBackup
end

function ColorConfiguratorDialog:setSliderValueWithNoCallback(sliderElement, value)
	local onChangedCallbackBackup = sliderElement.onChangedCallback

	sliderElement.onChangedCallback = NO_CALLBACK
	sliderElement:setValue(value)
	sliderElement.onChangedCallback = onChangedCallbackBackup
end

function ColorConfiguratorDialog:setPickerValuesWithNoCallback(pickerElement, value)
	local onChangedCallbackBackup = pickerElement.onChangedCallback

	pickerElement.onChangedCallback = NO_CALLBACK
	pickerElement:setValues(value)
	pickerElement.onChangedCallback = onChangedCallbackBackup
end

function ColorConfiguratorDialog:onIsUnicodeAllowedTextHue(unicode)
	local text, number = self:getTypedText(self.textInputHue, unicode)

	if number == nil or unicode == Input.KEY_space or unicode == Input.KEY_period or MathUtil.getIsOutOfBounds(number, 0, 360) or string.startsWith(text, "03") or string.startsWith(text, "02") or string.startsWith(text, "01") or string.startsWith(text, "00") then
		return false
	end

	return true
end

function ColorConfiguratorDialog:onIsUnicodeAllowedTextSatVal(element, unicode)
	local text, number = self:getTypedText(element, unicode)

	if number == nil or unicode == Input.KEY_space or unicode == Input.KEY_period or MathUtil.getIsOutOfBounds(number, 0, 100) or string.startsWith(text, "01") or string.startsWith(text, "00") then
		return false
	end

	return true
end

function ColorConfiguratorDialog:onIsUnicodeAllowedTextSaturation(unicode)
	return self:onIsUnicodeAllowedTextSatVal(self.textInputSaturation, unicode)
end

function ColorConfiguratorDialog:onIsUnicodeAllowedTextValue(unicode)
	return self:onIsUnicodeAllowedTextSatVal(self.textInputValue, unicode)
end

function ColorConfiguratorDialog:verifyInputTextHSV(textInputElement, maxValue, dismissal)
	local colorNumber = tonumber(textInputElement.text) or 0

	textInputElement:setText(tostring(MathUtil.clamp(colorNumber, 0, maxValue)))
end

function ColorConfiguratorDialog:onButtonPressedHue(element, dismissal)
	self:verifyInputTextHSV(element, 360, dismissal)
end

function ColorConfiguratorDialog:onButtonPressedSatVal(element, dismissal)
	self:verifyInputTextHSV(element, 100, dismissal)
end

function ColorConfiguratorDialog:updateHSVElements()
	local r, g, b = unpack(self.colors)
	local hsvColors = {ColorConfiguratorUtil.rgbToHsv(ColorConfiguratorUtil.giantsToRgb(r), ColorConfiguratorUtil.giantsToRgb(g), ColorConfiguratorUtil.giantsToRgb(b))}

	self:setPickerValuesWithNoCallback(self.colorPicker, hsvColors)

	for i, elem in pairs(self.textInputHSVElements) do
		self:setTextWithNoCallback(elem, tostring(hsvColors[i]))
	end
end

function ColorConfiguratorDialog:onChangedColorPicker(hsvColors, isSliderChange)
	self:onHSVElementChanged(hsvColors, false)
end

function ColorConfiguratorDialog:onTextChangedHue(element)
	local _, s, v = self.colorPicker:getValues()

	self:onHSVElementChanged({tonumber(element.text) or 0, s, v}, true)
end

function ColorConfiguratorDialog:onTextChangedSaturation(element)
	local h, _, v = self.colorPicker:getValues()

	self:onHSVElementChanged({h, tonumber(element.text) or 0, v}, true)
end

function ColorConfiguratorDialog:onTextChangedValue(element)
	local h, s, _ = self.colorPicker:getValues()

	self:onHSVElementChanged({h, s, tonumber(element.text) or 0}, true)
end

function ColorConfiguratorDialog:onHSVElementChanged(hsvColors, textChanged)
	if textChanged then
		self:setPickerValuesWithNoCallback(self.colorPicker, hsvColors)
	else
		self:playSample(self.sliderSample)

		for i, elem in pairs(self.textInputHSVElements) do
			self:setTextWithNoCallback(elem, tostring(hsvColors[i]))
		end
	end

	local r, g, b = ColorConfiguratorUtil.hsvToRgb(unpack(hsvColors))
	local giantsColors = {ColorConfiguratorUtil.rgbToGiants(r), ColorConfiguratorUtil.rgbToGiants(g), ColorConfiguratorUtil.rgbToGiants(b)}
	local elementsValues = {r, g, b}

	if not self.isRGBMode then
		elementsValues = giantsColors
	end

	for i = 1, 3 do
		self:setTextWithNoCallback(self.textInputElements[i], tostring(elementsValues[i]))
		self:setSliderValueWithNoCallback(self.sliderElements[i], elementsValues[i])

		self.colors[i] = giantsColors[i]
	end

	self:updateColorPreview()
	self:updateTooltipLayout(false)
end

function ColorConfiguratorDialog:onClickSelectColor()
	g_gui:showDialog("ColorConfiguratorListDialog")
end

function ColorConfiguratorDialog:getCurrentItemName(includeBrand)
	local name = ""
	local shopConfigScreen = self.pickerDialog.target

	if shopConfigScreen ~= nil then
		local storeItem = shopConfigScreen.storeItem

		if storeItem ~= nil then
			if includeBrand then
				name = g_brandManager:getBrandByIndex(storeItem.brandIndex).name .. " "
			end

			name = name .. storeItem.name
		end
	end

	return name
end

function ColorConfiguratorDialog:getIsMaterialChangeAllowed()
	local shopConfigScreen = self.pickerDialog.target
	local configName = self.pickerDialog.args.configName
	local vehicle = shopConfigScreen.previewVehicles[1] or shopConfigScreen.vehicle

	if vehicle ~= nil then
		local multipleItemPurchase = vehicle.spec_multipleItemPurchase
		local typeName = vehicle.typeName

		if multipleItemPurchase ~= nil and not multipleItemPurchase.isVehicle then
			return false
		end

		for config, types in pairs(self.disabledMaterialConfigs) do
			if configName == config then
				for _, type in pairs(types) do
					if typeName == type then
						return false
					end
				end
			end
		end
	end

	return true
end

function ColorConfiguratorDialog:getIsMaterialAvailable()
	local isEnabled = self:getIsMaterialChangeAllowed()

	if not isEnabled then
		local state = self:getMaterialState(0)

		if self.multiMaterial.state ~= state then
			self.multiMaterial:setState(state, true)
		end

		if FocusManager:getFocusedElement() == self.multiMaterial then
			FocusManager:setFocus(self.sliderRed.parent)
		end
	end

	self.multiMaterial:setDisabled(not isEnabled)

	return isEnabled
end

function ColorConfiguratorDialog:onFocusPicker(element)
	self:updateInputGlyphsVisibility(self.lastInputHelpMode == GS_INPUT_HELP_MODE_GAMEPAD)
	self.pickerSelected = true
end

function ColorConfiguratorDialog:onLeavePicker(element)
	self:updateInputGlyphsVisibility(false)
	self.pickerSelected = false
end