ColorConfigurator = {
	L10N_SYMBOL = {
		FIRST_RUN_INFO = "colorConfigurator_ui_info",
		CUSTOM_COLOR_NAME = "colorConfigurator_ui_customColorName"
	},
	MAX_TEMP_CONFIGS = 30
}

local ColorConfigurator_mt = Class(ColorConfigurator)

function ColorConfigurator.new(baseDir, settingsDir, modName, customMt)
	local self = setmetatable({}, customMt or ColorConfigurator_mt)

	addModEventListener(self)

	self.baseDirectory = baseDir
	self.customEnvironment = modName

	self.modSettings = {
		directory = settingsDir,
		baseConfigsXMLFilename = settingsDir .. "baseConfigs.xml",
		addConfigsXMLFilename = settingsDir .. "configs.xml",
		tempConfigsXMLFilename = settingsDir .. "tempConfigs.xml",
		settingsXMLFilename = settingsDir .. "settings.xml"
	}

	createFolder(self.modSettings.directory)
	deleteFile(self.modSettings.directory .. "initialized")

	self.settings = {
		initialized = false,
		isRGBMode = true,
		filterState = 1
	}

	self:loadSettingsFromXMLFile(true)

	self:addGuiElements()

	self.saveDialog = ColorConfiguratorSaveDialog.new(nil, nil, self)
	self.listDialog = ColorConfiguratorListDialog.new(nil, nil, self)
	self.pickerDialog = ColorConfiguratorPickerDialog.new(nil, nil, self)
	self.dialog = ColorConfiguratorDialog.new(nil, nil, Utils.getFilename("materials.xml", baseDir), self)

	ColorConfiguratorUtil.overwrittenFunction(GuiOverlay, "loadOverlay", self, "guiOverlay_loadOverlay")

	g_gui:loadProfiles(Utils.getFilename("gui/guiProfiles.xml", baseDir))

	local currentGui = FocusManager.currentGui

	g_gui:loadGui(Utils.getFilename("gui/ColorConfiguratorSaveDialog.xml", baseDir), "ColorConfiguratorSaveDialog", self.saveDialog)
	g_gui:loadGui(Utils.getFilename("gui/ColorConfiguratorListDialog.xml", baseDir), "ColorConfiguratorListDialog", self.listDialog)
	g_gui:loadGui(g_gui.guis["ColorPickerDialog"].xmlFilename, "ColorConfiguratorPickerDialog", self.pickerDialog)
	g_gui:loadGui(Utils.getFilename("gui/ColorConfiguratorDialog.xml", baseDir), "ColorConfiguratorDialog", self.dialog)

	if currentGui ~= nil then
		FocusManager:setGui(currentGui)
	end

	self.supportedConfigurations = {
		"baseMaterial",
		"designMaterial",
		"baseColor",
		"designColor",
		"wrappingColor",
		"rimColor"
	}

	for i = 2, 49 do
		table.insert(self.supportedConfigurations, "designMaterial" .. i)
	end

	self.baseConfigs = self:loadBaseConfigs(true)
	self.additionalConfigs = self:loadAdditionalConfigs(true)
	self.tempConfigs = self:loadTempConfigs(true)

	addConsoleCommand("ccRemoveBaseConfigs", "", "consoleCommandRemoveBaseConfigs", self)
	addConsoleCommand("ccRemoveVehiclesConfigs", "", "consoleCommandRemoveVehiclesConfigs", self)
	addConsoleCommand("ccRemoveTempConfigs", "", "consoleCommandRemoveTempConfigs", self)

	ColorConfiguratorUtil.overwrittenFunction(g_gui, "showColorPickerDialog", self, "g_gui_showColorPickerDialog")
	ColorConfiguratorUtil.prependedFunction(TypeManager, "finalizeTypes", self, "typeManager_finalizeTypes")
	ColorConfiguratorUtil.appendedFunction(ConfigurationUtil, "getConfigColorPostLoad", self, "configurationUtil_getConfigColorPostLoad")
	ColorConfiguratorUtil.overwrittenFunction(ConfigurationUtil, "getMaterialByConfigId", self, "configurationUtil_getMaterialByConfigId", true)
	ColorConfiguratorUtil.overwrittenFunction(ConfigurationUtil, "getColorByConfigId", self, "configurationUtil_getColorByConfigId", true)
	ColorConfiguratorUtil.overwrittenFunction(ConfigurationUtil, "applyConfigMaterials", self, "configurationUtil_applyConfigMaterials", true)
	ColorConfiguratorUtil.overwrittenFunction(LicensePlateManager, "readLicensePlateData", self, "licensePlateManager_readLicensePlateData", true)
	ColorConfiguratorUtil.appendedFunction(LicensePlateManager, "writeLicensePlateData", self, "licensePlateManager_writeLicensePlateData")
	ColorConfiguratorUtil.prependedFunction(VehicleLoadingUtil, "loadVehiclesAtPlaceStepFinished", self, "vehicleLoadingUtil_loadVehiclesAtPlaceStepFinished")
	ColorConfiguratorUtil.appendedFunction(ShopConfigScreen, "onPickColor", self, "shopConfigScreen_onPickColor")
	ColorConfiguratorUtil.overwrittenFunction(ShopConfigScreen, "onChangeLicensePlate", self, "shopConfigScreen_onChangeLicensePlate")
	ColorConfiguratorUtil.overwrittenFunction(ShopConfigScreen, "getConfigurationCostsAndChanges", self, "shopConfigScreen_getConfigurationCostsAndChanges")
	ColorConfiguratorUtil.overwrittenFunction(ShopConfigScreen, "setStoreItem", self, "shopConfigScreen_setStoreItem")
	ColorConfiguratorUtil.overwrittenFunction(ShopConfigScreen, "onVehicleLoaded", self, "shopConfigScreen_onVehicleLoaded")
	ColorConfiguratorUtil.prependedFunction(ShopConfigScreen, "updateData", self, "shopConfigScreen_updateData")
	ColorConfiguratorUtil.overwrittenFunction(WorkshopScreen, "setConfigurations", self, "workshopScreen_setConfigurations")
	ColorConfiguratorUtil.overwrittenFunction(ChangeVehicleConfigEvent, "run", self, "changeVehicleConfigEvent_run")
	ColorConfiguratorUtil.overwrittenFunction(VehicleSaleSystem, "generateRandomVehicle", self, "vehicleSaleSystem_generateRandomVehicle")

	return self
end

function ColorConfigurator:addGuiElements()
	local mapping = Gui.CONFIGURATION_CLASS_MAPPING

	mapping.colorConfiguratorRender = ColorConfiguratorRenderElement
	mapping.colorConfiguratorSlider = ColorConfiguratorSliderElement
	mapping.colorConfiguratorPicker = ColorConfiguratorPickerElement
	mapping.colorConfiguratorMultiOption = ColorConfiguratorMultiOptionElement
	mapping.colorConfiguratorTextInput = ColorConfiguratorTextInputElement

	local procFuncs = Gui.ELEMENT_PROCESSING_FUNCTIONS

	procFuncs.colorConfiguratorMultiOption = Gui.assignPlaySampleCallback
	procFuncs.colorConfiguratorTextInput = Gui.assignPlaySampleCallback
end

function ColorConfigurator:getSetting(name)
	return self.settings[name]
end

function ColorConfigurator:updateSetting(name, value, doNotSave)
	if self.settings[name] ~= nil and self.settings[name] ~= value then
		self.settings[name] = value

		if not doNotSave then
			self:saveSettingsToXMLFile()
		end

		return true
	end

	return false
end

function ColorConfigurator:saveSettingsToXMLFile(printToLog)
	local xmlFile = XMLFile.create("settingsXML", self.modSettings.settingsXMLFilename, "settings")

	if xmlFile ~= nil then
		local settings = self.settings

		xmlFile:setBool("settings.init", settings.initialized)
		xmlFile:setBool("settings.rgb", settings.isRGBMode)
		xmlFile:setInt("settings.filter", settings.filterState)

		xmlFile:save()
		xmlFile:delete()

		ColorConfiguratorUtil.printOnSave(printToLog, xmlFile)
	end
end

function ColorConfigurator:loadSettingsFromXMLFile(printToLog)
	local xmlFile = XMLFile.loadIfExists("settingsXML", self.modSettings.settingsXMLFilename, "settings")

	if xmlFile ~= nil then
		local settings = self.settings

		settings.initialized = xmlFile:getBool("settings.init", settings.initialized)
		settings.isRGBMode = xmlFile:getBool("settings.rgb", settings.isRGBMode)
		settings.filterState = xmlFile:getInt("settings.filter", settings.filterState)

		xmlFile:delete()

		ColorConfiguratorUtil.printOnLoad(printToLog, xmlFile)
	end
end

function ColorConfigurator:loadMap(filename)
	self.listDialog:onMapLoad()
	self.dialog:onMapLoad()
end

function ColorConfigurator:saveBaseConfigs(printToLog)
	local xmlFile = XMLFile.create("baseConfigsXML", self.modSettings.baseConfigsXMLFilename, "configs")

	if xmlFile ~= nil then
		xmlFile:setTable("configs.config", self.baseConfigs, function(key, config, _)
			xmlFile:setString(key .. "#name", config.name)
			xmlFile:setVector(key .. "#color", config.color)
			xmlFile:setInt(key .. "#material", config.material)
		end)

		xmlFile:save()
		xmlFile:delete()

		ColorConfiguratorUtil.printOnSave(printToLog, xmlFile)
	end
end

function ColorConfigurator:loadBaseConfigs(printToLog)
	local baseConfigs = {}
	local xmlFile = XMLFile.loadIfExists("baseConfigsXML", self.modSettings.baseConfigsXMLFilename, "configs")

	if xmlFile ~= nil then
		xmlFile:iterate("configs.config", function (_, configKey)
			local name = xmlFile:getString(configKey .. "#name")
			local color = xmlFile:getVector(configKey .. "#color")
			local material = xmlFile:getInt(configKey .. "#material")

			table.insert(baseConfigs, {name = name, color = color, material = material})
		end)

		xmlFile:delete()

		ColorConfiguratorUtil.printOnLoad(printToLog, xmlFile)
	end

	return baseConfigs
end

function ColorConfigurator:addBaseConfig(configData, doNotSave)
	if configData == nil then
		return false
	end

	table.insert(self.baseConfigs, configData)

	if not doNotSave then
		self:saveBaseConfigs()
	end

	self.listDialog:updateListData()

	return true
end

function ColorConfigurator:removeBaseConfig(configData, doNotSave)
	if configData == nil then
		return false
	end

	if table.removeElement(self.baseConfigs, configData) then
		if not doNotSave then
			self:saveBaseConfigs()
		end

		return true
	end

	return false
end

function ColorConfigurator:getBaseConfigs()
	return self.baseConfigs
end

function ColorConfigurator:consoleCommandRemoveBaseConfigs(verbose)
	verbose = Utils.stringToBoolean(verbose)

	local ret = "Use 'true' parameter to remove all base configurations."

	if verbose then
		table.clear(self.baseConfigs)
		self:saveBaseConfigs()

		self.listDialog:updateListData()

		ret = "All configurations removed."
	end

	return ret
end

function ColorConfigurator:saveAdditionalConfigs(printToLog)
	local xmlFile = XMLFile.create("configsXML", self.modSettings.addConfigsXMLFilename, "vehicles")

	if xmlFile ~= nil then
		xmlFile:setTable("vehicles.vehicle", self.additionalConfigs, function(vehicleKey, vehicles, vehicleName)
			xmlFile:setString(vehicleKey .. "#filename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(vehicleName)))

			xmlFile:setTable(vehicleKey .. ".configs", vehicles, function(configsKey, configs, configName)
				xmlFile:setString(configsKey .. "#name", configName)

				xmlFile:setTable(configsKey .. ".config", configs, function(configKey, config, _)
					xmlFile:setString(configKey .. "#name", config.name)
					xmlFile:setVector(configKey .. "#color", config.color)
					xmlFile:setInt(configKey .. "#material", config.material)
				end)
			end)
		end)

		xmlFile:save()
		xmlFile:delete()

		ColorConfiguratorUtil.printOnSave(printToLog, xmlFile)
	end
end

function ColorConfigurator:loadAdditionalConfigs(printToLog)
	local additionalConfigs = {}
	local xmlFile = XMLFile.loadIfExists("configsXML", self.modSettings.addConfigsXMLFilename, "vehicles")

	if xmlFile ~= nil then
		xmlFile:iterate("vehicles.vehicle", function (_, vehicleKey)
			local filename = ColorConfiguratorUtil.convertFromNetworkFilename(xmlFile:getString(vehicleKey .. "#filename"))

			if additionalConfigs[filename] == nil then
				additionalConfigs[filename] = {}
			end

			xmlFile:iterate(vehicleKey .. ".configs", function (_, configsKey)
				local configName = xmlFile:getString(configsKey .. "#name")

				if additionalConfigs[filename][configName] == nil then
					additionalConfigs[filename][configName] = {}
				end

				xmlFile:iterate(configsKey .. ".config", function (_, configKey)
					local name = xmlFile:getString(configKey .. "#name")
					local color = xmlFile:getVector(configKey .. "#color")
					local material = xmlFile:getInt(configKey .. "#material")

					table.insert(additionalConfigs[filename][configName], {
						name = name,
						color = color,
						material = material,
						isAddConfig = true
					})
				end)
			end)
		end)

		xmlFile:delete()

		ColorConfiguratorUtil.printOnLoad(printToLog, xmlFile)
	end

	return additionalConfigs
end

function ColorConfigurator:addAdditionalConfig(filename, configName, configData, doNotSave)
	if filename == nil or configName == nil or configData == nil then
		return false
	end

	if self.additionalConfigs[filename] == nil then
		self.additionalConfigs[filename] = {}
	end

	if self.additionalConfigs[filename][configName] == nil then
		self.additionalConfigs[filename][configName] = {}
	end

	table.insert(self.additionalConfigs[filename][configName], configData)

	if not doNotSave then
		self:saveAdditionalConfigs()
	end

	return true
end

function ColorConfigurator:removeAdditionalConfig(filename, configName, configData, doNotSave)
	if not self:getIsAddConfigExists(filename, configName) or configData == nil then
		return false
	end

	if table.removeElement(self.additionalConfigs[filename][configName], configData) then
		if next(self.additionalConfigs[filename][configName]) == nil then
			self.additionalConfigs[filename][configName] = nil

			if next(self.additionalConfigs[filename]) == nil then
				self.additionalConfigs[filename] = nil
			end
		end

		if not doNotSave then
			self:saveAdditionalConfigs()
		end

		return true
	end

	return false
end

function ColorConfigurator:getAdditionalConfigs(filename, configName)
	if not self:getIsAddConfigExists(filename, configName) then
		return
	end

	return self.additionalConfigs[filename][configName]
end

function ColorConfigurator:getIsAddConfigExists(filename, configName)
	return filename ~= nil and configName ~= nil and self.additionalConfigs[filename] ~= nil and self.additionalConfigs[filename][configName] ~= nil
end

function ColorConfigurator:consoleCommandRemoveVehiclesConfigs(verbose)
	verbose = Utils.stringToBoolean(verbose)

	local ret = "Use 'true' parameter to remove all vehicles configurations."

	if verbose then
		g_gui:closeDialogByName("ColorConfiguratorPickerDialog")

		table.clear(self.additionalConfigs)
		self:saveAdditionalConfigs()

		ret = "All configurations removed."
	end

	return ret
end

function ColorConfigurator:saveTempConfigs(printToLog)
	local xmlFile = XMLFile.create("tempConfigsXML", self.modSettings.tempConfigsXMLFilename, "configs")

	if xmlFile ~= nil then
		xmlFile:setTable("configs.config", self.tempConfigs, function(key, config, _)
			xmlFile:setString(key .. "#name", config.name)
			xmlFile:setString(key .. "#date", config.date)
			xmlFile:setVector(key .. "#color", config.color)
			xmlFile:setInt(key .. "#material", config.material)
		end)

		xmlFile:save()
		xmlFile:delete()

		ColorConfiguratorUtil.printOnSave(printToLog, xmlFile)
	end
end

function ColorConfigurator:loadTempConfigs(printToLog)
	local tempConfigs = {}
	local xmlFile = XMLFile.loadIfExists("tempConfigsXML", self.modSettings.tempConfigsXMLFilename, "configs")

	if xmlFile ~= nil then
		xmlFile:iterate("configs.config", function (_, configKey)
			local name = xmlFile:getString(configKey .. "#name")
			local date = xmlFile:getString(configKey .. "#date", "n/a")
			local color = xmlFile:getVector(configKey .. "#color")
			local material = xmlFile:getInt(configKey .. "#material")

			table.insert(tempConfigs, {name = name, date = date, color = color, material = material})
		end)

		xmlFile:delete()

		ColorConfiguratorUtil.printOnLoad(printToLog, xmlFile)
	end

	return tempConfigs
end

function ColorConfigurator:addTempConfig(colors, name, doNotSave)
	local color = {colors[1], colors[2], colors[3]}
	local material = colors[4]
	local date = getDate("%H:%M:%S")

	local tempConfigs = table.copy(self.tempConfigs, math.huge)

	for i, temp in pairs(tempConfigs) do
		if temp.material == material and ColorConfiguratorUtil.equals(temp.color, color) and temp.name == name then
			table.remove(tempConfigs, i)
		end
	end

	table.insert(tempConfigs, {
		name = name,
		date = date,
		color = color,
		material = material
	})

	local x = #tempConfigs - ColorConfigurator.MAX_TEMP_CONFIGS
	local newTempConfigs = {}

	if x > 0 then
		for i = x + 1, #tempConfigs do
			table.push(newTempConfigs, tempConfigs[i])
		end
	else
		newTempConfigs = tempConfigs
	end

	self.tempConfigs = newTempConfigs
	self.listDialog:updateListData()

	if not doNotSave then
		self:saveTempConfigs()
	end
end

function ColorConfigurator:getTempConfigs()
	return self.tempConfigs
end

function ColorConfigurator:consoleCommandRemoveTempConfigs(verbose)
	verbose = Utils.stringToBoolean(verbose)

	local ret = "Use 'true' parameter to remove all temp configurations."

	if verbose then
		table.clear(self.tempConfigs)
		self:saveTempConfigs()

		self.listDialog:updateListData()

		ret = "All configurations removed."
	end

	return ret
end

function ColorConfigurator:onFirstTimeRun()
	local mod = g_modManager:getModByName(self.customEnvironment)
	local text = string.format("%s\n\n%s", utf8ToUpper(mod.title), g_i18n:getText(ColorConfigurator.L10N_SYMBOL.FIRST_RUN_INFO))

	g_gui:showInfoDialog({
		dialogType = DialogElement.TYPE_INFO,
		text = text
	})
end

function ColorConfigurator:setColor(colors)
	local shopConfigScreen = self.pickerDialog.target
	local configName = self.pickerDialog.args.configName
	local colorOptionIndex = self.pickerDialog.args.colorOptionIndex

	if g_gui.currentGuiName ~= "ShopConfigScreen" or shopConfigScreen == nil or configName == nil or colorOptionIndex == nil or colors == nil then
		Logging.error("Cannot set color for target configuration")
		return
	end

	local configIndex = nil

	for _, configItem in pairs(shopConfigScreen.storeItem.configurations[configName]) do
		if configItem.isCustomConfig then
			configIndex = configItem.index
			break
		end
	end

	if configIndex == nil then
		Logging.error("Cannot find custom configuration index '%s' for '%s'", configName, shopConfigScreen.storeItem.xmlFilename)
		return
	end

	if shopConfigScreen.licensePlateData == nil then
		shopConfigScreen.licensePlateData = {}
	end

	if shopConfigScreen.licensePlateData.colorConfiguratorData == nil then
		shopConfigScreen.licensePlateData.colorConfiguratorData = {}
	end

	shopConfigScreen.licensePlateData.colorConfiguratorData[configName] = {
		color = {colors[1], colors[2], colors[3]},
		material = colors[4]
	}

	shopConfigScreen:onPickColor(configIndex, {
		configName = configName,
		colorOptionIndex = colorOptionIndex
	}, false)
end

function ColorConfigurator:g_gui_showColorPickerDialog(gui, superFunc, args)
	local guiName = "ColorPickerDialog"

	if args ~= nil and args.args ~= nil and args.target == g_shopConfigScreen and args.callback == g_shopConfigScreen.onPickColor then
		for _, configName in pairs(self.supportedConfigurations) do
			if configName == args.args.configName then
				guiName = "ColorConfiguratorPickerDialog"
				break
			end
		end
	end

	local dialog = gui:showDialog(guiName)

	if dialog ~= nil and args ~= nil then
		dialog.target:setCallback(args.callback, args.target, args.args)
		dialog.target:setColors(args.colors, args.defaultColor, args.defaultColorMaterial)
	end

	if guiName == "ColorConfiguratorPickerDialog" then
		if self:updateSetting("initialized", true) then
			self:onFirstTimeRun()
		end
	end
end

function ColorConfigurator:typeManager_finalizeTypes(typeManager)
	if typeManager.typeName == "vehicle" then
		for typeName, typeEntry in pairs(typeManager:getTypes()) do
			typeManager:addSpecialization(typeName, self.customEnvironment .. ".vehicleColor")
		end
	end
end

function ColorConfigurator:configurationUtil_getConfigColorPostLoad(xmlFile, baseKey, baseDir, customEnvironment, isMod, configurationItems, storeItem)
	local isSelectable = false

	for _, item in ipairs(configurationItems) do
		if item.isSelectable ~= false then
			isSelectable = true
			break
		end
	end

	if isSelectable then
		local isSupported = false

		for _, configName in pairs(self.supportedConfigurations) do
			if baseKey == string.format("vehicle.%sConfigurations", configName) then
				isSupported = true
				break
			end
		end

		if isSupported then
			local configItem = StoreItemUtil.addConfigurationItem(configurationItems, g_i18n:getText(ColorConfigurator.L10N_SYMBOL.CUSTOM_COLOR_NAME), "", 0, 0, false)

			configItem.color = {1, 0, 1, 0}
			configItem.isCustomConfig = true
		end
	end
end

function ColorConfigurator:configurationUtil_getMaterialByConfigId(superFunc, object, configName, configId)
	if configId ~= nil and object.getCustomConfigurationByName ~= nil then
		local customConfig = object:getCustomConfigurationByName(configName)

		if customConfig ~= nil then
			return customConfig.material
		end
	end

	return superFunc(object, configName, configId)
end

function ColorConfigurator:configurationUtil_getColorByConfigId(superFunc, object, configName, configId)
	if configId ~= nil and object.getCustomConfigurationByName ~= nil then
		local customConfig = object:getCustomConfigurationByName(configName)

		if customConfig ~= nil then
			local r, g, b = unpack(customConfig.color)

			return {
				r,
				g,
				b,
				customConfig.material
			}
		end
	end

	return superFunc(object, configName, configId)
end

function ColorConfigurator:configurationUtil_applyConfigMaterials(superFunc, object, xmlFile, configName, configId)
	local customConfigFound = false

	if configId ~= nil and object.getCustomConfigurationByName ~= nil then
		local customConfig = object:getCustomConfigurationByName(configName)

		if customConfig ~= nil then
			local customConfigFound = true
			local storeItem = g_storeManager:getItemByXMLFilename(object.configFileName)

			if storeItem ~= nil then
				local defaultConfigId = StoreItemUtil.getDefaultConfigId(storeItem, configName)
				local configuration = g_configurationManager:getConfigurationDescByName(configName)
				local xmlKey = configuration.xmlKey

				if xmlKey ~= nil then
					xmlKey = "." .. xmlKey
				else
					xmlKey = ""
				end

				local configKey = string.format("vehicle%s.%sConfigurations.%sConfiguration(%d)", xmlKey, configName, configName, defaultConfigId - 1)

				if xmlFile:hasProperty(configKey) then
					xmlFile:iterate(configKey .. ".material", function (_, key)
						local baseMaterialNode = xmlFile:getValue(key .. "#node", nil, object.components, object.i3dMappings)
						local refMaterialNode = xmlFile:getValue(key .. "#refNode", nil, object.components, object.i3dMappings)

						if baseMaterialNode ~= nil and refMaterialNode ~= nil then
							local oldMaterial = getMaterial(baseMaterialNode, 0)
							local newMaterial = getMaterial(refMaterialNode, 0)

							for _, component in pairs(object.components) do
								ConfigurationUtil.replaceMaterialRec(object, component.node, oldMaterial, newMaterial)
							end
						end

						local materialName = xmlFile:getValue(key .. "#name")

						if materialName ~= nil then
							local shaderParameterName = xmlFile:getValue(key .. "#shaderParameter")

							if shaderParameterName ~= nil then
								if object.setBaseMaterialColor ~= nil then
									object:setBaseMaterialColor(materialName, shaderParameterName, customConfig.color, customConfig.material)
								end
							end
						end
					end)
				end
			end
		end
	end

	if not customConfigFound then
		superFunc(object, xmlFile, configName, configId)
	end
end

function ColorConfigurator:licensePlateManager_readLicensePlateData(superFunc, streamId, connection)
	local licensePlateData = superFunc(streamId, connection)
	local hasColorConfiguratorData = streamReadBool(streamId)

	if hasColorConfiguratorData then
		local numConfigurations = streamReadUInt8(streamId)

		licensePlateData.colorConfiguratorData = {}

		for _ = 1, numConfigurations do
			local configName = streamReadString(streamId)
			local r = streamReadFloat32(streamId)
			local g = streamReadFloat32(streamId)
			local b = streamReadFloat32(streamId)
			local material = streamReadInt32(streamId)

			licensePlateData.colorConfiguratorData[configName] = {
				color = ColorConfiguratorUtil.formatColor(r, g, b),
				material = material
			}
		end
	end

	return licensePlateData
end

function ColorConfigurator:licensePlateManager_writeLicensePlateData(streamId, connection, licensePlateData)
	if streamWriteBool(streamId, licensePlateData ~= nil and licensePlateData.colorConfiguratorData ~= nil) then
		local numConfigurations = 0

		for _, _ in pairs(licensePlateData.colorConfiguratorData) do
			numConfigurations = numConfigurations + 1
		end

		streamWriteUInt8(streamId, numConfigurations)

		for configName, configData in pairs(licensePlateData.colorConfiguratorData) do
			streamWriteString(streamId, configName)
			streamWriteFloat32(streamId, configData.color[1])
			streamWriteFloat32(streamId, configData.color[2])
			streamWriteFloat32(streamId, configData.color[3])
			streamWriteInt32(streamId, configData.material)
		end
	end
end

function ColorConfigurator:vehicleLoadingUtil_loadVehiclesAtPlaceStepFinished(_, vehicle, vehicleLoadState, arguments)
	local vehiclesToLoad = arguments.vehiclesToLoad

	if vehicle ~= nil and vehiclesToLoad.licensePlateData ~= nil and vehicle.setCustomConfigurationsData ~= nil then
		vehicle:setCustomConfigurationsData(vehiclesToLoad.licensePlateData.colorConfiguratorData)
	end
end

function ColorConfigurator:shopConfigScreen_onPickColor(shopConfigScreen, colorIndex, args, noUpdate)
	if colorIndex ~= nil then
		local configName = args.configName
		local isCustomConfig = shopConfigScreen.storeItem.configurations[configName][colorIndex].isCustomConfig

		if isCustomConfig then
			local colorConfiguratorData = shopConfigScreen.licensePlateData.colorConfiguratorData

			if colorConfiguratorData ~= nil then
				local customConfig = colorConfiguratorData[configName]

				if customConfig ~= nil then
					local colorOptionIndex = args.colorOptionIndex
					local element = shopConfigScreen.colorElements[colorOptionIndex]
					local color = customConfig.color
					local material = customConfig.material
					local isMetallic = ConfigurationUtil.isColorMetallic(material)

					element:getDescendantByName("colorImage"):setImageColor(nil, unpack(color))
					element:getDescendantByName("colorImage"):setVisible(not isMetallic)
					element:getDescendantByName("colorImageMetallic"):setImageColor(nil, unpack(color))
					element:getDescendantByName("colorImageMetallic"):setVisible(isMetallic)
				end
			end
		end
	end
end

function ColorConfigurator:shopConfigScreen_onChangeLicensePlate(shopConfigScreen, superFunc, licensePlateData)
	if licensePlateData ~= nil and shopConfigScreen.licensePlateData.colorConfiguratorData ~= nil then
		licensePlateData.colorConfiguratorData = shopConfigScreen.licensePlateData.colorConfiguratorData
	end

	superFunc(shopConfigScreen, licensePlateData)
end

function ColorConfigurator:shopConfigScreen_getConfigurationCostsAndChanges(shopConfigScreen, superFunc, storeItem, vehicle, saleItem)
	local basePrice, upgradePrice, hasChanges = superFunc(shopConfigScreen, storeItem, vehicle, saleItem)

	if not hasChanges and vehicle ~= nil then
		if shopConfigScreen.licensePlateData ~= nil then
			if vehicle.getCustomConfigurationsDataIsEqual ~= nil then
				if not vehicle:getCustomConfigurationsDataIsEqual(shopConfigScreen.licensePlateData.colorConfiguratorData) then
					hasChanges = true
				end
			end
		end
	end

	return basePrice, upgradePrice, hasChanges
end

function ColorConfigurator:shopConfigScreen_setStoreItem(shopConfigScreen, superFunc, storeItem, vehicle, saleItem, configBasePrice, configurations)
	shopConfigScreen:deletePreviewVehicles()

	shopConfigScreen.storeItem = storeItem
	shopConfigScreen.vehicle = vehicle
	shopConfigScreen.saleItem = saleItem
	shopConfigScreen.configBasePrice = Utils.getNoNil(configBasePrice, 0)

	if configurations == nil then
		configurations = {}

		if storeItem.defaultConfigurationIds ~= nil then
			for configName, index in pairs(storeItem.defaultConfigurationIds) do
				configurations[configName] = index
			end
		end

		if vehicle ~= nil and vehicle.configurations ~= nil then
			for configName, index in pairs(vehicle.configurations) do
				configurations[configName] = index
			end
		end

		if saleItem ~= nil and saleItem.boughtConfigurations ~= nil then
			for configName, boughtItems in pairs(saleItem.boughtConfigurations) do
				for index, value in pairs(boughtItems) do
					if value then
						configurations[configName] = index
					end
				end
			end
		end
	end

	shopConfigScreen.configurations = configurations
	shopConfigScreen.subConfigurations = {}
	shopConfigScreen.currentConfigSet = 1
	shopConfigScreen.isLoadingInitial = true
	shopConfigScreen.licensePlateData = {}

	if g_licensePlateManager:getAreLicensePlatesAvailable() then
		if vehicle ~= nil then
			if vehicle.getLicensePlatesData ~= nil then
				local data = vehicle:getLicensePlatesData()

				if data ~= nil and data.characters ~= nil then
					shopConfigScreen.licensePlateData = {
						variation = data.variation,
						colorIndex = data.colorIndex,
						placementIndex = data.placementIndex,
						characters = table.copy(data.characters)
					}
					local defaultPlacementIndex, hasFrontPlate = vehicle:getLicensePlateDialogSettings()
					shopConfigScreen.licensePlateData.defaultPlacementIndex = defaultPlacementIndex
					shopConfigScreen.licensePlateData.hasFrontPlate = hasFrontPlate
				else
					shopConfigScreen.licensePlateData = g_currentMission:getLastCreatedLicensePlate() or g_licensePlateManager:getRandomLicensePlateData()
				end
			end
		else
			shopConfigScreen.licensePlateData = g_currentMission:getLastCreatedLicensePlate() or g_licensePlateManager:getRandomLicensePlateData()
		end
	end

	if vehicle ~= nil and vehicle.getCustomConfigurations ~= nil then
		local colorConfiguratorData = vehicle:getCustomConfigurations()

		if colorConfiguratorData ~= nil then
			shopConfigScreen.licensePlateData.colorConfiguratorData = table.copy(colorConfiguratorData, math.huge)
		end
	end

	shopConfigScreen:processStoreItemConfigurations(storeItem, vehicle, saleItem)
	shopConfigScreen:updateDisplay(storeItem, vehicle, saleItem)
	shopConfigScreen:resetCamera()
end

function ColorConfigurator:shopConfigScreen_onVehicleLoaded(shopConfigScreen, superFunc, vehicle, loadingState, asyncArguments)
	if loadingState == VehicleLoadingUtil.VEHICLE_LOAD_OK then
		if asyncArguments.openCounter == shopConfigScreen.openCounter then
			if shopConfigScreen.isLoadingInitial or shopConfigScreen.isOpen then
				if shopConfigScreen.licensePlateData ~= nil and vehicle.setCustomConfigurationsData ~= nil then
					vehicle:setCustomConfigurationsData(shopConfigScreen.licensePlateData.colorConfiguratorData)
				end
			end
		end
	end

	superFunc(shopConfigScreen, vehicle, loadingState, asyncArguments)
end

function ColorConfigurator:shopConfigScreen_updateData(shopConfigScreen, storeItem, vehicle, saleItem)
	if shopConfigScreen.licensePlateData ~= nil and shopConfigScreen.licensePlateData.colorConfiguratorData ~= nil then
		for configName, _ in pairs(shopConfigScreen.licensePlateData.colorConfiguratorData) do
			local configId = shopConfigScreen.configurations[configName]
			local config = storeItem.configurations[configName][configId]

			if config ~= nil and not config.isCustomConfig then
				shopConfigScreen.licensePlateData.colorConfiguratorData[configName] = nil
			end
		end
	end
end

function ColorConfigurator:workshopScreen_setConfigurations(workshopScreen, superFunc, vehicle, buyItem, storeItem, configs, price, licensePlateData)
	workshopScreen.vehicle = vehicle

	workshopScreen.shopConfigScreen:onClickBack()

	if not buyItem and storeItem ~= nil and configs ~= nil then
		local areChangesMade = false
		local newConfigs = {}

		for configName, configValue in pairs(configs) do
			if workshopScreen.vehicle.configurations[configName] ~= configValue then
				newConfigs[configName] = configs[configName]
				areChangesMade = true
			end
		end

		if workshopScreen.vehicle.getLicensePlatesDataIsEqual ~= nil and not workshopScreen.vehicle:getLicensePlatesDataIsEqual(licensePlateData) then
			areChangesMade = true
		end

		if licensePlateData ~= nil and workshopScreen.vehicle.getCustomConfigurationsDataIsEqual ~= nil and not workshopScreen.vehicle:getCustomConfigurationsDataIsEqual(licensePlateData.colorConfiguratorData) then
			areChangesMade = true
		end

		if areChangesMade then
			if not g_currentMission.controlPlayer and g_currentMission.controlledVehicle ~= nil and workshopScreen.vehicle == g_currentMission.controlledVehicle then
				g_currentMission:onLeaveVehicle()
			end

			g_client:getServerConnection():sendEvent(ChangeVehicleConfigEvent.new(workshopScreen.vehicle, price, g_currentMission:getFarmId(), newConfigs, licensePlateData))
		end
	else
		workshopScreen:onClickBack()

		if workshopScreen.owner ~= nil then
			workshopScreen.owner:openMenu()
		end
	end
end

function ColorConfigurator:changeVehicleConfigEvent_run(changeVehicleConfigEvent, superFunc, connection)
	if not connection:getIsServer() then
		local success = false
		local vehicle = changeVehicleConfigEvent.vehicle

		if vehicle ~= nil and vehicle.isVehicleSaved and not vehicle.isControlled and g_currentMission:getHasPlayerPermission("buyVehicle", connection) then
			for configName, configId in pairs(changeVehicleConfigEvent.configurations) do
				ConfigurationUtil.addBoughtConfiguration(vehicle, configName, configId)
				ConfigurationUtil.setConfiguration(vehicle, configName, configId)
			end

			vehicle:setLicensePlatesData(changeVehicleConfigEvent.licensePlateData)

			if changeVehicleConfigEvent.licensePlateData ~= nil and vehicle.setCustomConfigurationsData ~= nil then
				vehicle:setCustomConfigurationsData(changeVehicleConfigEvent.licensePlateData.colorConfiguratorData)
			end

			vehicle.isReconfigurating = true

			g_server:broadcastEvent(VehicleSetIsReconfiguratingEvent.new(vehicle), nil, nil, vehicle)

			local xmlFile = Vehicle.getReloadXML(vehicle)
			local key = "vehicles.vehicle(0)"

			local function asyncCallbackFunction(_, newVehicle, vehicleLoadState, arguments)
				if vehicleLoadState == VehicleLoadingUtil.VEHICLE_LOAD_OK then
					success = true

					g_currentMission:addMoney(-changeVehicleConfigEvent.price, changeVehicleConfigEvent.farmId, MoneyType.SHOP_VEHICLE_BUY, true)
					vehicle:removeFromPhysics()
					g_currentMission:removeVehicle(vehicle)
				else
					g_currentMission:removeVehicle(newVehicle)
					vehicle:addToPhysics()
				end

				xmlFile:delete()
				connection:sendEvent(ChangeVehicleConfigEvent.newServerToClient(success))
			end

			VehicleLoadingUtil.loadVehicleFromSavegameXML(xmlFile, key, false, false, nil, nil, asyncCallbackFunction, nil, {})
		else
			connection:sendEvent(ChangeVehicleConfigEvent.newServerToClient(false))
		end

		return
	end

	g_workshopScreen:onVehicleChanged(changeVehicleConfigEvent.successful)
end

function ColorConfigurator:guiOverlay_loadOverlay(guiOverlay, superFunc, overlay, overlayName, imageSize, profile, xmlFile, key)
	local overlay = superFunc(guiOverlay, overlay, overlayName, imageSize, profile, xmlFile, key)

	if overlay ~= nil and overlay.filename == "g_colorConfiguratorUIFilename" then
		overlay.filename = Utils.getFilename("gui/ui_elements.png", self.baseDirectory)
	end

	return overlay
end

function ColorConfigurator:vehicleSaleSystem_generateRandomVehicle(saleSystem, superFunc)
	local saleItem = superFunc(saleSystem)

	if saleItem ~= nil then
		local recalculatePrice = false
		local storeItem = g_storeManager:getItemByXMLFilename(saleItem.xmlFilename)

		for configName, boughtItems in pairs(saleItem.boughtConfigurations) do
			for index, value in pairs(boughtItems) do
				if value then
					if storeItem.configurations[configName][index].isCustomConfig then
						saleItem.boughtConfigurations[configName][index] = nil

						local defaultConfigId = StoreItemUtil.getDefaultConfigId(storeItem, configName)

						saleItem.boughtConfigurations[configName][defaultConfigId] = true
						recalculatePrice = true
					end
				end
			end
		end

		if recalculatePrice then
			local age = math.random(6, 40)
			local damage = math.random() * 0.4 + 0.2
			local wear = math.random() * 0.8 + 0.2
			local operatingTime = age * (math.random() * 0.8 + 0.5) * 60 * 60 * 1000
			local defaultPrice = StoreItemUtil.getDefaultPrice(storeItem, saleItem.boughtConfigurations)
			local repairPrice = Wearable.calculateRepairPrice(defaultPrice, damage, age)
			local repaintPrice = Wearable.calculateRepaintPrice(defaultPrice, wear)
			local price = Vehicle.calculateSellPrice(storeItem, age, operatingTime, defaultPrice, repairPrice, repaintPrice)

			saleItem.age = age
			saleItem.price = price
			saleItem.damage = damage
			saleItem.wear = wear
			saleItem.operatingTime = operatingTime
		end
	end

	return saleItem
end

g_colorConfigurator = ColorConfigurator.new(g_currentModDirectory, g_currentModSettingsDirectory, g_currentModName)